<?php
/**
 * Plugin Name: Image Sitemap (Root XML)
 * Plugin URI: https://www.jan-siefken.com
 * Description: Erstellt automatisch eine Bilder-Sitemap (image-sitemap.xml) im WordPress-Rootverzeichnis.
 * Version: 1.0.0
 * Author: Jan Siefken
 * Author URI: https://www.jan-siefken.com
 *
 * Haftungsausschluss:
 * Dieses Plugin wird ohne Gewähr bereitgestellt. Der Einsatz erfolgt auf eigene Verantwortung.
 * Für Datenverluste, Fehlfunktionen oder Schäden, die durch die Nutzung entstehen,
 * wird keine Haftung übernommen.
 */

if (!defined('ABSPATH')) {
    exit;
}

class ISR_Image_Sitemap_Root {

    const CRON_HOOK = 'isr_generate_image_sitemap_daily';
    const FILENAME  = 'image-sitemap.xml';

    public static function init() {
        register_activation_hook(__FILE__, [__CLASS__, 'on_activate']);
        register_deactivation_hook(__FILE__, [__CLASS__, 'on_deactivate']);

        add_action(self::CRON_HOOK, [__CLASS__, 'generate_sitemap']);

        add_action('admin_menu', [__CLASS__, 'admin_menu']);
        add_action('admin_post_isr_generate_now', [__CLASS__, 'handle_generate_now']);
    }

    public static function on_activate() {
        if (!wp_next_scheduled(self::CRON_HOOK)) {
            wp_schedule_event(time() + 60, 'daily', self::CRON_HOOK);
        }
        self::generate_sitemap();
    }

    public static function on_deactivate() {
        $timestamp = wp_next_scheduled(self::CRON_HOOK);
        if ($timestamp) {
            wp_unschedule_event($timestamp, self::CRON_HOOK);
        }
    }

    public static function admin_menu() {
        add_management_page(
            'Image Sitemap',
            'Image Sitemap',
            'manage_options',
            'isr-image-sitemap',
            [__CLASS__, 'admin_page']
        );
    }

    public static function admin_page() {
        if (!current_user_can('manage_options')) {
            return;
        }

        $path     = self::get_target_path();
        $url      = home_url('/' . self::FILENAME);
        $writable = is_writable(ABSPATH);
        ?>
        <div class="wrap">
            <h1>Image Sitemap (Root XML)</h1>

            <p><strong>Datei:</strong> <code><?php echo esc_html($path); ?></code></p>
            <p><strong>URL:</strong>
                <a href="<?php echo esc_url($url); ?>" target="_blank" rel="noopener">
                    <?php echo esc_html($url); ?>
                </a>
            </p>

            <p>
                <strong>Root schreibbar:</strong>
                <?php echo $writable
                    ? '<span style="color:green;">Ja</span>'
                    : '<span style="color:red;">Nein</span>'; ?>
            </p>

            <form method="post" action="<?php echo esc_url(admin_url('admin-post.php')); ?>">
                <?php wp_nonce_field('isr_generate_now'); ?>
                <input type="hidden" name="action" value="isr_generate_now">
                <?php submit_button('Sitemap jetzt generieren'); ?>
            </form>

            <p style="margin-top:20px;color:#666;">
                Die Sitemap wird täglich automatisch per WP-Cron aktualisiert.
            </p>
        </div>
        <?php
    }

    public static function handle_generate_now() {
        if (!current_user_can('manage_options')) {
            wp_die('No permission.');
        }

        check_admin_referer('isr_generate_now');
        self::generate_sitemap();

        wp_safe_redirect(admin_url('tools.php?page=isr-image-sitemap'));
        exit;
    }

    private static function get_target_path() {
        return trailingslashit(ABSPATH) . self::FILENAME;
    }

    public static function generate_sitemap() {
        $items = self::collect_image_items();
        $xml   = self::build_xml($items);

        return (bool) @file_put_contents(self::get_target_path(), $xml);
    }

    private static function collect_image_items() {
        $items = [];

        $query = new WP_Query([
            'post_type'      => ['post', 'page'],
            'post_status'    => 'publish',
            'posts_per_page' => -1,
            'fields'         => 'ids',
            'no_found_rows'  => true,
        ]);

        foreach ($query->posts as $post_id) {
            $permalink = get_permalink($post_id);
            if (!$permalink) {
                continue;
            }

            $images = [];

            // Featured Image
            $thumb_id = get_post_thumbnail_id($post_id);
            if ($thumb_id) {
                $src = wp_get_attachment_image_url($thumb_id, 'full');
                if ($src) {
                    $images[] = $src;
                }
            }

            // Content Images
            $content = get_post_field('post_content', $post_id);
            if ($content) {
                preg_match_all('/<img[^>]+src=["\']([^"\']+)["\']/i', $content, $matches);
                if (!empty($matches[1])) {
                    $images = array_merge($images, $matches[1]);
                }
            }

            $images = self::normalize_and_filter_images($images);

            if ($images) {
                $items[] = [
                    'loc'     => $permalink,
                    'lastmod' => get_post_modified_time('c', true, $post_id),
                    'images'  => $images,
                ];
            }
        }

        wp_reset_postdata();
        return $items;
    }

    /**
     * Fix A: Erzwingt HTTPS
     * Fix B: Nur Bilder der eigenen Domain
     */
    private static function normalize_and_filter_images(array $images) {
        $out    = [];
        $myHost = wp_parse_url(home_url('/'), PHP_URL_HOST);

        foreach ($images as $img) {
            if (!$img) {
                continue;
            }

            // relativ → absolut
            if (strpos($img, '//') === 0) {
                $img = 'https:' . $img;
            } elseif (!preg_match('#^https?://#i', $img)) {
                $img = home_url($img);
            }

            // https erzwingen
            $img = preg_replace('#^http://#i', 'https://', $img);

            // nur eigene Domain
            $host = wp_parse_url($img, PHP_URL_HOST);
            if ($host && $myHost && strcasecmp($host, $myHost) !== 0) {
                continue;
            }

            $path = wp_parse_url($img, PHP_URL_PATH);
            if (!$path) {
                continue;
            }

            $ext = strtolower(pathinfo($path, PATHINFO_EXTENSION));
            if (!in_array($ext, ['jpg', 'jpeg', 'png', 'gif', 'webp', 'svg'], true)) {
                continue;
            }

            $out[] = esc_url_raw($img);
        }

        return array_values(array_unique($out));
    }

    private static function build_xml(array $items) {
        $xml  = '<?xml version="1.0" encoding="UTF-8"?>' . "\n";
        $xml .= '<urlset xmlns="http://www.sitemaps.org/schemas/sitemap/0.9" ';
        $xml .= 'xmlns:image="http://www.google.com/schemas/sitemap-image/1.1">' . "\n";

        foreach ($items as $item) {
            $xml .= "  <url>\n";
            $xml .= '    <loc>' . esc_xml($item['loc']) . "</loc>\n";
            $xml .= '    <lastmod>' . esc_xml($item['lastmod']) . "</lastmod>\n";

            foreach ($item['images'] as $img) {
                $xml .= "    <image:image>\n";
                $xml .= '      <image:loc>' . esc_xml($img) . "</image:loc>\n";
                $xml .= "    </image:image>\n";
            }

            $xml .= "  </url>\n";
        }

        $xml .= "</urlset>\n";
        return $xml;
    }
}

// esc_xml Fallback
if (!function_exists('esc_xml')) {
    function esc_xml($text) {
        return htmlspecialchars((string) $text, ENT_XML1 | ENT_COMPAT, 'UTF-8');
    }
}

ISR_Image_Sitemap_Root::init();

/**
 * Readme-Link in der Plugin-Übersicht (Plugins → Installierte Plugins)
 */
add_filter(
    'plugin_action_links_' . plugin_basename(__FILE__),
    function ($links) {
        $links[] = '<a href="' . esc_url(plugins_url('readme.txt', __FILE__)) . '" target="_blank" rel="noopener">Readme</a>';
        return $links;
    }
);
